//------------------------------------------------------------------------------
//! @file WiMOD_SAP_LORAWAN_IDs.h
//! @ingroup WiMOD_SAP_LORAWAN
//! <!------------------------------------------------------------------------->
//! @brief Supporting IDs and definitions for the LoRaWAN ServiceAccessPoint
//! @version 0.1
//! <!------------------------------------------------------------------------->
//!
//! 
//!
//! <!--------------------------------------------------------------------------
//! Copyright (c) 2016
//! IMST GmbH
//! Carl-Friedrich Gauss Str. 2-4
//! 47475 Kamp-Lintfort
//! --------------------------------------------------------------------------->
//! @author (FB), IMST
//! <!--------------------------------------------------------------------------
//! Target OS:    none
//! Target CPU:   tbd
//! Compiler:     tbd
//! --------------------------------------------------------------------------->
//! @internal
//! @par Revision History:
//! <PRE>
//!-----------------------------------------------------------------------------
//! Version | Date       | Author | Comment
//!-----------------------------------------------------------------------------
//!
//! </PRE>
//------------------------------------------------------------------------------


/*
 * THIS IS AN EXAMPLE IMPLEMENTATION ACCORDING THE THE HCI SPEC: V1.12
 * FOR FIRMWARE: LoRaWAN_EndNode_Modem
 *
 * SEE FILE: WiMOD_LoRaWAN_EndNode_Modem_HCI_Spec_V1_12.pdf
 * for detailed information
 *
 */


#ifndef ARDUINO_WIMOD_SAP_LORAWAN_IDS_H_
#define ARDUINO_WIMOD_SAP_LORAWAN_IDS_H_

//------------------------------------------------------------------------------
//
// Section Includes Files
//
//------------------------------------------------------------------------------

#include "utils/WMDefs.h"

//------------------------------------------------------------------------------
//
// Service Access Point Identifier
//
//------------------------------------------------------------------------------

/** SAP ID for the LoRaWAN service */
#define LORAWAN_SAP_ID                      0x10



//------------------------------------------------------------------------------
//
//  LoRaWAN Message Identifier
//
//------------------------------------------------------------------------------
//! @cond Doxygen_Suppress

// Status Codes

#define LORAWAN_STATUS_OK                       0x00 //Operation successful
#define LORAWAN_STATUS_ERROR                    0x01 //Operation failed
#define LORAWAN_STATUS_CMD_NOT_SUPPORTED        0x02 //Command is not supported
#define LORAWAN_STATUS_WRONG_PARAMETER          0x03 //HCI message contains wrong parameter
#define LORAWAN_STATUS_WRONG_DEVICE_MODE        0x04 //Stack is running in a wrong mode
#define LORAWAN_STATUS_DEVICE_NOT_ACTIVATED     0x05 //Device is not activated
#define LORAWAN_STATUS_DEVICE_BUSY              0x06 //Device is busy, command rejected
#define LORAWAN_STATUS_QUEUE_FULL               0x07 //Message queue is full, command rejected
#define LORAWAN_STATUS_LENGTH_ERROR             0x08 //HCI message length is invalid or radio payload size is too large
#define LORAWAN_STATUS_NO_FACTORY_SETTINGS      0x09 //Factory Settings EEPROM block missing
#define LORAWAN_STATUS_CHANNEL_BLOCKED          0x0A //Channel blocked by Duty Cycle
#define LORAWAN_STATUS_CHANNEL_NOT AVAILABLE    0x0B //No channel available (e.g. no channel defined for the configured spreading factor)

//! @endcond

/**
 * @brief Status code enumeration for LoRaWAN SAP of the WiMOD EndNode Modem firmware
 */
typedef enum TLoRaWANStatus
{
    LoRaWAN_Status_Ok = 0,                                                      /*!< OK; No error */
    LoRaWAN_Status_Error,                                                       /*!< general (un-specific) error */
    LoRaWAN_Status_Cmd_Not_Supported,                                           /*!< Command is not supported (or unknown) */
    LoRaWAN_Status_Wrong_parameter,                                             /*!< Wrong parameter applied */
    LoRaWAN_Status_Wrong_DeviceMode,                                            /*!< Device is in wong device mode for specific command */
    LoRaWAN_Status_Device_Not_Activated,                                        /*!< Device has NOT been activated; command has not been executed */
    LoRaWAN_Status_Device_Busy,                                                 /*!< There is another operation pending; and or duty cycle restriction is active  */
    LoRaWAN_Status_Queue_Full,                                                  /*!< internal command queue is full */
    LoRaWAN_Status_Length_Error,                                                /*!< some error related to the given length information */
    LoRaWAN_Status_No_Factory_Settings,                                         /*!< Factory settings are not available */
    LoRaWAN_Status_Channel_Blocked,                                             /*!< Selected channel is blocked (duty cycle?) */
    LoRaWAN_Status_Channel_Not_Available,                                       /*!< Selected channel is not available */
} TLoRaWANStatus;

//! @cond Doxygen_Suppress

#define LORAWAN_FORMAT_STATUS_JOIN_OK                0x00
#define LORAWAN_FORMAT_STATUS_JOIN_OK_EXT_HCI        0x01


// Status Format Codes
//  This is a bit mask field!
#define LORAWAN_FORMAT_OK                       0x00
#define LORAWAN_FORMAT_EXT_HCI_OUT_ACTIVE       0x01
#define LORAWAN_FORMAT_ACK_RECEIVED             0x02
#define LORAWAN_FORMAT_FP_RECEIVED              0x04

// Message IDs
#define LORAWAN_MSG_ACTIVATE_DEVICE_REQ         0x01
#define LORAWAN_MSG_ACTIVATE_DEVICE_RSP         0x02
#define LORAWAN_MSG_SET_JOIN_PARAM_REQ          0x05
#define LORAWAN_MSG_SET_JOIN_PARAM_RSP          0x06
#define LORAWAN_MSG_JOIN_NETWORK_REQ            0x09
#define LORAWAN_MSG_JOIN_NETWORK_RSP            0x0A
#define LORAWAN_MSG_JOIN_NETWORK_TX_IND         0x0B
#define LORAWAN_MSG_JOIN_NETWORK_IND            0x0C
#define LORAWAN_MSG_SEND_UDATA_REQ              0x0D
#define LORAWAN_MSG_SEND_UDATA_RSP              0x0E
#define LORAWAN_MSG_SEND_UDATA_TX_IND           0x0F
#define LORAWAN_MSG_RECV_UDATA_IND              0x10
#define LORAWAN_MSG_SEND_CDATA_REQ              0x11
#define LORAWAN_MSG_SEND_CDATA_RSP              0x12
#define LORAWAN_MSG_SEND_CDATA_TX_IND           0x13
#define LORAWAN_MSG_RECV_CDATA_IND              0x14
#define LORAWAN_MSG_RECV_ACK_IND                0x15
#define LORAWAN_MSG_RECV_NO_DATA_IND            0x16
#define LORAWAN_MSG_SET_RSTACK_CONFIG_REQ       0x19
#define LORAWAN_MSG_SET_RSTACK_CONFIG_RSP       0x1A
#define LORAWAN_MSG_GET_RSTACK_CONFIG_REQ       0x1B
#define LORAWAN_MSG_GET_RSTACK_CONFIG_RSP       0x1C
#define LORAWAN_MSG_REACTIVATE_DEVICE_REQ       0x1D
#define LORAWAN_MSG_REACTIVATE_DEVICE_RSP       0x1E
#define LORAWAN_MSG_DEACTIVATE_DEVICE_REQ       0x21
#define LORAWAN_MSG_DEACTIVATE_DEVICE_RSP       0x22
#define LORAWAN_MSG_FACTORY_RESET_REQ           0x23
#define LORAWAN_MSG_FACTORY_RESET_RSP           0x24
#define LORAWAN_MSG_SET_DEVICE_EUI_REQ          0x25
#define LORAWAN_MSG_SET_DEVICE_EUI_RSP          0x26
#define LORAWAN_MSG_GET_DEVICE_EUI_REQ          0x27
#define LORAWAN_MSG_GET_DEVICE_EUI_RSP          0x28
#define LORAWAN_MSG_GET_NWK_STATUS_REQ          0x29
#define LORAWAN_MSG_GET_NWK_STATUS_RSP          0x2A
#define LORAWAN_MSG_SEND_MAC_CMD_REQ            0x2B
#define LORAWAN_MSG_SEND_MAC_CMD_RSP            0x2C
#define LORAWAN_MSG_RECV_MAC_CMD_IND            0x2D
#define LORAWAN_MSG_SET_CUSTOM_CFG_REQ          0x31
#define LORAWAN_MSG_SET_CUSTOM_CFG_RSP          0x32
#define LORAWAN_MSG_GET_CUSTOM_CFG_REQ          0x33
#define LORAWAN_MSG_GET_CUSTOM_CFG_RSP          0x34



// Radio Stack Config Options; bit field

#define LORAWAN_STK_OPTION_ADR                  (0x01 << 0)
#define LORAWAN_STK_OPTION_DUTY_CYCLE_CTRL      (0x01 << 1)
#define LORAWAN_STK_OPTION_DEV_CLASS_C          (0x01 << 2)
#define LORAWAN_STK_OPTION_EXT_PKT_FORMAT       (0x01 << 6)
#define LORAWAN_STK_OPTION_MAC_CMD              (0x01 << 7)


// LoRa Data Rate Index

#define LORAWAN_DATA_RATE_LORA_SF12_125KHZ      0
#define LORAWAN_DATA_RATE_LORA_SF11_125KHZ      1
#define LORAWAN_DATA_RATE_LORA_SF10_125KHZ      2
#define LORAWAN_DATA_RATE_LORA_SF9_125KHZ       3
#define LORAWAN_DATA_RATE_LORA_SF8_125KHZ       4
#define LORAWAN_DATA_RATE_LORA_SF7_125KHZ       5
#define LORAWAN_DATA_RATE_LORA_SF7_250KHZ       6
#define LORAWAN_DATA_RATE_LORA_FSK              7

#define IS_LORAWAN_DATA_RATE(x)                 ( ((x) >= LORAWAN_DATA_RATE_LORA_SF12_125KHZ) && \
                                                  ((X) <= LORAWAN_DATA_RATE_LORA_FSK) )

//! @endcond

/**
 * @brief LoRa data rate definitions
 */
typedef enum TLoRaWANDataRate
{
    LoRaWAN_DataRate_LoRa_SF12_125kHz = 0,                                      /*!< LoRa Spreading factor 12 using 125kHz bandwidth */
    LoRaWAN_DataRate_LoRa_SF11_125kHz,                                          /*!< LoRa Spreading factor 11 using 125kHz bandwidth */
    LoRaWAN_DataRate_LoRa_SF10_125kHz,                                          /*!< LoRa Spreading factor 10 using 125kHz bandwidth */
    LoRaWAN_DataRate_LoRa_SF9_125kHz,                                           /*!< LoRa Spreading factor  9 using 125kHz bandwidth */
    LoRaWAN_DataRate_LoRa_SF8_125kHz,                                           /*!< LoRa Spreading factor  8 using 125kHz bandwidth */
    LoRaWAN_DataRate_LoRa_SF7_125kHz,                                           /*!< LoRa Spreading factor  7 using 125kHz bandwidth */
    LoRaWAN_DataRate_LoRa_SF7_250kHz,                                           /*!< LoRa Spreading factor  7 using 250kHz bandwidth */
    LoRaWAN_DataRate_LoRa_FSK,                                                  /*!< FSK modulation; LoRa settings not applicable    */
} TLoRaWANDataRate;

//! @cond Doxygen_Suppress

// LoRa Channel Index ( EU868 )

#define LORAWAN_CH_EU_868_1_MHZ                    0
#define LORAWAN_CH_EU_868_3_MHZ                    1
#define LORAWAN_CH_EU_868_5_MHZ                    2
#define LORAWAN_CH_EU_869_525_MHZ                  128

#define IS_LORAWAN_CH_EU(x)                       ( ((x) == LORAWAN_CH_EU_868_1_MHZ) || \
                                                  ((x) == LORAWAN_CH_EU_868_3_MHZ) || \
                                                  ((x) == LORAWAN_CH_EU_868_5_MHZ) || \
                                                  ((x) == LORAWAN_CH_EU_869_525_MHZ) )

//! @endcond

/**
 * @brief Channel definition for EU 868MHz Band
 */
typedef enum TLoRaWAN_Channel_EU
{
    LoRaWAN_Channel_EU_868_1_Mhz   = 0,                                         /*!< carrier frequency of 868.100 MHz */
    LoRaWAN_Channel_EU_868_3_Mhz   = 1,                                         /*!< carrier frequency of 868.300 MHz */
    LoRaWAN_Channel_EU_868_5_Mhz   = 2,                                         /*!< carrier frequency of 868.500 MHz */
    LoRaWAN_Channel_EU_869_525_Mhz = 128,                                       /*!< carrier frequency of 869.525 MHz */
} TLoRaWAN_Channel_EU;


//! @cond Doxygen_Suppress

// LoRa Channel Index ( India )

#define LORAWAN_CH_IN_865_062_5_MHZ                0
#define LORAWAN_CH_IN_865_402_5_MHZ                1
#define LORAWAN_CH_IN_868_5_MHZ                    2
#define LORAWAN_CH_IN_866_550_MHZ                  128

#define IS_LORAWAN_CH_IN(x)                       ( ((x) == LORAWAN_CH_IN_865_062_5_MHZ) || \
                                                  ((x) == LORAWAN_CH_IN_865_402_5_MHZ) || \
                                                  ((x) == LORAWAN_CH_IN_868_5_MHZ) || \
                                                  ((x) == LORAWAN_CH_IN_866_550_MHZ) )
//! @endcond


/**
 * @brief Channel definition for India
 */
typedef enum TLoRaWAN_Channel_India
{
    LoRaWAN_Channel_India_865_062_5_Mhz   = 0,                                  /*!< carrier frequency of 865.062 500 MHz */
    LoRaWAN_Channel_India_865_402_5_Mhz   = 1,                                  /*!< carrier frequency of 865.402 500 MHz */
    LoRaWAN_Channel_India_868_5_Mhz       = 2,                                  /*!< carrier frequency of 868.500 MHz */
    LoRaWAN_Channel_India_866_550_Mhz     = 128,                                /*!< carrier frequency of 866.550 MHz */
} TLoRaWAN_Channel_India;

//! @cond Doxygen_Suppress

// @deprecated removed in HCI spec V1_15

//// LoRa Channel Index ( New Zealand )
//
//#define LORAWAN_CH_NZ_865_0_MHZ                    0
//#define LORAWAN_CH_NZ_865_2_MHZ                    1
//#define LORAWAN_CH_NZ_865_4_MHZ                    2
//#define LORAWAN_CH_NZ_866_2_MHZ                    3
//#define LORAWAN_CH_NZ_866_4_MHZ                    4
//#define LORAWAN_CH_NZ_866_6_MHZ                    5
//#define LORAWAN_CH_NZ_866_8_MHZ                    6
//#define LORAWAN_CH_NZ_867_0_MHZ                    7
//#define LORAWAN_CH_NZ_865_6_MHZ                    8
//#define LORAWAN_CH_NZ_865_9_MHZ                    9
//#define LORAWAN_CH_NZ_867_2_MHZ                    128
//
/*
//#define IS_LORAWAN_CH_NZ(x)                       ( ((x) == LORAWAN_CH_NZ_865_0_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_865_2_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_865_4_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_866_2_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_866_4_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_865_4_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_866_6_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_867_0_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_865_6_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_865_9_MHZ) || \
//                                                  ((x) == LORAWAN_CH_NZ_867_2_MHZ) )
*/
//
//! @endcond

// @deprecated removed in HCI spec V1_15
///**
// * @brief Channel definition for New Zealand
// */
//typedef enum TLoRaWAN_Channel_NewZealand
//{
//    LoRaWAN_Channel_NewZealand_865_0_Mhz   = 0,                                 /*!< carrier frequency of 865.000 MHz */
//    LoRaWAN_Channel_NewZealand_865_2_Mhz   = 1,                                 /*!< carrier frequency of 866.200 MHz */
//    LoRaWAN_Channel_NewZealand_865_4_Mhz   = 2,                                 /*!< carrier frequency of 865.400 MHz */
//    LoRaWAN_Channel_NewZealand_866_2_Mhz   = 3,                                 /*!< carrier frequency of 866.200 MHz */
//    LoRaWAN_Channel_NewZealand_866_4_Mhz   = 4,                                 /*!< carrier frequency of 866.400 MHz */
//    LoRaWAN_Channel_NewZealand_866_6_Mhz   = 5,                                 /*!< carrier frequency of 866.600 MHz */
//    LoRaWAN_Channel_NewZealand_866_8_Mhz   = 6,                                 /*!< carrier frequency of 866.800 MHz */
//    LoRaWAN_Channel_NewZealand_867_0_Mhz   = 7,                                 /*!< carrier frequency of 867.000 MHz */
//    LoRaWAN_Channel_NewZealand_865_6_Mhz   = 8,                                 /*!< carrier frequency of 865.600 MHz */
//    LoRaWAN_Channel_NewZealand_865_9_Mhz   = 9,                                 /*!< carrier frequency of 865.900 MHz */
//    LoRaWAN_Channel_NewZealand_867_2_Mhz   = 128,                               /*!< carrier frequency of 867.200 MHz */
//} TLoRaWAN_Channel_NewZealand;


//! @cond Doxygen_Suppress

// @deprecated removed in HCI spec V1_15
//// LoRa Channel Index ( Singapore )
//
//#define LORAWAN_CH_SG_923_5_MHZ                    0
//#define LORAWAN_CH_SG_923_7_MHZ                    1
//#define LORAWAN_CH_SG_923_9_MHZ                    2
//#define LORAWAN_CH_SG_924_1_MHZ                    3
//#define LORAWAN_CH_SG_924_3_MHZ                    4
//#define LORAWAN_CH_SG_924_5_MHZ                    5
//#define LORAWAN_CH_SG_924_7_MHZ                    6
//#define LORAWAN_CH_SG_924_9_MHZ                    7
//#define LORAWAN_CH_SG_923_8_MHZ                    8
//#define LORAWAN_CH_SG_922_5_MHZ                    128
//
/*
//#define IS_LORAWAN_CH_SG(x)                       ( ((x) == LORAWAN_CH_SG_923_5_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_923_7_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_923_9_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_924_1_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_924_3_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_924_5_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_924_7_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_924_9_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_923_8_MHZ) || \
//                                                  ((x) == LORAWAN_CH_SG_922_5_MHZ) )
*/
//! @endcond

// @deprecated removed in HCI spec V1_15
///**
// * @brief Channel definition for Singapore
// */
//typedef enum TLoRaWAN_Channel_Singapore
//{
//    LoRaWAN_Channel_Singapore_923_5_Mhz   = 0,                                  /*!< carrier frequency of 923.500 MHz */
//    LoRaWAN_Channel_Singapore_923_7_Mhz   = 1,                                  /*!< carrier frequency of 823.700 MHz */
//    LoRaWAN_Channel_Singapore_923_9_Mhz   = 2,                                  /*!< carrier frequency of 923.900 MHz */
//    LoRaWAN_Channel_Singapore_924_1_Mhz   = 3,                                  /*!< carrier frequency of 924.100 MHz */
//    LoRaWAN_Channel_Singapore_924_3_Mhz   = 4,                                  /*!< carrier frequency of 924.300 MHz */
//    LoRaWAN_Channel_Singapore_924_5_Mhz   = 5,                                  /*!< carrier frequency of 924.500 MHz */
//    LoRaWAN_Channel_Singapore_924_7_Mhz   = 6,                                  /*!< carrier frequency of 924.700 MHz */
//    LoRaWAN_Channel_Singapore_924_9_Mhz   = 7,                                  /*!< carrier frequency of 924.900 MHz */
//    LoRaWAN_Channel_Singapore_923_8_Mhz   = 8,                                  /*!< carrier frequency of 923.800 MHz */
//    LoRaWAN_Channel_Singapore_922_5_Mhz   = 128,                                /*!< carrier frequency of 922.500 MHz */
//} TLoRaWAN_Channel_Singapore;
//

//! @cond Doxygen_Suppress

// LoRa Radio Band Index

#define LORAWAN_BAND_EU_868                     1
#define LORAWAN_BAND_RESERVED                   2
#define LORAWAN_BAND_IN_865                     3
//#define LORAWAN_BAND_NZ_865                     4	// @deprecated removed in HCI spec V1_15
//#define LORAWAN_BAND_SG_923                     5 // @deprecated removed in HCI spec V1_15
#define LORAWAN_BAND_EU_868_RX2_SF9             129

#define IS_LORAWAN_BAND(x)                      ( ((x) == LORAWAN_BAND_EU_868) || \
                                                  ((x) == LORAWAN_BAND_IN_865) || \
                                                  ((x) == LORAWAN_BAND_NZ_865) || \
                                                  ((x) == LORAWAN_BAND_SG_923) || \
                                                  ((x) == LORAWAN_BAND_EU_868_RX2_SF9) )
//! @endcond

/**
 * @brief Channel Band Selection
 */
typedef enum TLoRaWAN_FreqBand
{
    LoRaWAN_FreqBand_EU_868          = 1,                                       /*!< default EU 868 MHz band */
    LoRaWAN_FreqBand_Reserved        = 2,                                       /*!< reserved */
    LoRaWAN_FreqBand_India_865       = 3,                                       /*!< India */
//    LoRaWAN_FreqBand_NewZealand_865  = 4,                                       /*!< NewZealand */  // @deprecated removed in HCI spec V1_15
//    LoRaWAN_FreqBand_Singapore_923   = 5,                                       /*!< Singapore */   // @deprecated removed in HCI spec V1_15
    LoRaWAN_FreqBand_EU_868_RX2_SF9  = 129,                                     /*!< alternative EU band, using SF9 for 2nd RX window */
} TLoRaWAN_FreqBand;


//! @cond Doxygen_Suppress
// LoRa Radio Power Level

#define LORAWAN_TX_POWER_LEVEL_MIN                  0
#define LORAWAN_TX_POWER_LEVEL_MAX                  20

#define IS_LORAWAN_TX_POWER_LEVEL(x)            ( ((x) >= LORAWAN_TX_POWER_LEVEL_MIN) && \
                                                  ((X) <= LORAWAN_TX_POWER_LEVEL_MAX) )


// LoRa Radio Power Saving Mode

#define LORAWAN_POWER_SAVING_MODE_OFF               0
#define LORAWAN_POWER_SAVING_MODE_AUTO              1

//! @endcond

/**
 * @brief Power Saving modes
 */
typedef enum TLoRaWAN_PowerSaving
{
    LoRaWAN_PowerSaving_Off = 0,                                                /*!<Power Saving is NOT used (=off) */
    LoRaWAN_PowerSaving_On  = 1,                                                /*!< automatic Power Saving is enabled (=on) */
} TLoRaWAN_PowerSaving;


//! @cond Doxygen_Suppress
// LoRaWAN Network Status

#define LORAWAN_NWK_SATUS_INACTIVE                  0
#define LORAWAN_NWK_STATUS_ACTIVE_ABP               1
#define LORAWAN_NWK_STATUS_ACTIVE_OTAA              2
#define LORAWAN_NWK_STATUS_JOINING_OTAA             3

//! @endcond

/**
 * @brief Network Status Status defines
 */
typedef enum TLoRaWAN_NwkStatus
{
    LoRaWAN_NwkStatus_Inactive     = 0,                                         /*!< Device is not active ( = no Data transmission is allowed) */
    LoRaWAN_NwkStatus_Active_ABP   = 1,                                         /*!< Device has been activated / registered via ABP procedure */
    LoRaWAN_NwkStatus_Active_OTAA  = 2,                                         /*!< Device has been activated / registered via OTAA procedure */
    LoRaWAN_NwkStatus_Joining_OTAA = 3,                                         /*!< OTAA procedure is currently in progress */
} TLoRaWAN_NwkStatus;

//! @cond Doxygen_Suppress

// LoRaWAN MAC Commands

#define LORAWAN_MAC_DATA_SERVICE_TYPE_U_DATA        0
#define LORAWAN_MAC_DATA_SERVICE_TYPE_C_DATA        1

//! @endcond

//------------------------------------------------------------------------------
//
// misc. defines
//
//------------------------------------------------------------------------------
//! @cond Doxygen_Suppress

#define WiMODLORAWAN_NWK_SESSION_KEY_LEN            16      // 128 bit
#define WiMODLORAWAN_APP_SESSION_KEY_LEN            16      // 128 bit

#define WiMODLORAWAN_DEV_EUI_LEN                    8       // 64 bit
#define WiMODLORAWAN_APP_EUI_LEN                    8       // 64 bit
#define WiMODLORAWAN_APP_KEY_LEN                    16      // 128 bit

#define WiMODLORAWAN_MAC_CMD_PAYLOAD_LENGTH         30

//! @endcond

//------------------------------------------------------------------------------
//
// structures
//
//------------------------------------------------------------------------------

/**
 * @brief Activation data structure for LoRa network activation
 */
typedef struct TWiMODLORAWAN_ActivateDeviceData
{
    UINT32      DeviceAddress;                                                  /*!< address of the WiMOD within the LoRa network (pre-set for ABP; set by nwk if OTAA  was used*/
    UINT8       NwkSKey[WiMODLORAWAN_NWK_SESSION_KEY_LEN];                      /*!< secret network session key (128bit) */
    UINT8       AppSKey[WiMODLORAWAN_NWK_SESSION_KEY_LEN];                      /*!< secret application session key (128bit) */
} TWiMODLORAWAN_ActivateDeviceData;


/**
 * @brief JOIN data structure for LoRa network activation
 */
typedef struct TWiMODLORAWAN_JoinParams
{
    UINT8       AppEUI[WiMODLORAWAN_APP_EUI_LEN];                               /*!< application EUI (64bit) */
    UINT8       AppKey[WiMODLORAWAN_APP_KEY_LEN];                               /*!< application key (128it) */
} TWiMODLORAWAN_JoinParams;


/**
 * @brief Optional TX information flag
 */
typedef enum TWiMOD_OptIndInfos
{
    LORAWAN_OPT_TX_IND_INFOS_NOT_AVAILABLE = 0,                                 /*!< no TX informations attached / available */
    LORAWAN_OPT_TX_IND_INFOS_EXCL_PKT_CNT  = 1,                                 /*!< ext. infos available, but without packet counter */
    LORAWAN_OPT_TX_IND_INFOS_INCL_PKT_CNT  = 2,                                 /*!< ext. infos available inclusive packet counter */
} TWiMOD_OptIndInfos;



/**
 * @brief Activation data structure for LoRa network activation
 */
typedef struct TWiMODLORAWAN_TxIndData
{
    TWiMOD_OptIndInfos      FieldAvailability;                                  /*!< indicator flag for optional infos */

    UINT8                   StatusFormat;                                       /*!< Status/Format field of reponse packet from WiMOD */
    // optional - see StatusFormat code
    UINT8                   ChannelIndex;                                       /*!< used channel index (@see TLoRaWAN_Channel_* definitions)  */
    // optional - see StatusFormat code
    UINT8                   DataRateIndex;                                      /*!< used data rate index (@see TLoRaWANDataRate)  */
    // optional - see StatusFormat code
    UINT8                   NumTxPackets;                                       /*!< number of TX packet (for current operation; can be used as retransmission counter) */
    // new in V1.14 optional
    UINT8					PowerLevel;											/*!< transmit power in dBm */
    // new in V1.14 optional
    UINT32					RfMsgAirtime;										/*!< airtime in ms of packet transmission */
} TWiMODLORAWAN_TxIndData;

/*
 * Number of payload bytes for a user defined message
 *
 * Note: the actual number of payload bytes can be less, if the air frame
 * contains a list of MAC commands.
 */
//! @cond Doxygen_Suppress
#define WiMODLORAWAN_APP_PAYLOAD_LEN                128
//! @endcond

/**
 * @brief Structure containing all data necessary to transmitting a data message
 */
typedef struct TWiMODLORAWAN_TX_Data
{
    UINT8       Port;                                                           /*!< target LoRaWAN port number */
    UINT8       Length;                                                         /*!< Length of the payload data */
    UINT8       Payload[WiMODLORAWAN_APP_PAYLOAD_LEN];                          /*!< (user) payload */
} TWiMODLORAWAN_TX_Data;


/**
 * @brief Structure containing RX (user) payload data
 */
typedef struct TWiMODLORAWAN_RX_Data
{
    UINT8       StatusFormat;                                                   /*!< Status/Format Field for this message */
    UINT8       Port;                                                           /*!< (Target) LoRaWAN port */
    UINT8       Length;                                                         /*!< Length of the used (user) payload field */
    UINT8       Payload[WiMODLORAWAN_APP_PAYLOAD_LEN];                          /*!< payload data */

    // optional RX info
    UINT8       ChannelIndex;                                                   /*!< used channel index (@see TLoRaWAN_Channel_* definitions)*/
    UINT8       DataRateIndex;                                                  /*!< used data rate index (@see TLoRaWANDataRate)  */
    INT8        RSSI;                                                           /*!< RSSI value for the received packet */
    INT8        SNR;                                                            /*!< SNR value for the received packet  */
    UINT8       RxSlot;                                                         /*!< number of the rx slot that contained the messsage */
    bool        OptionalInfoAvaiable;                                           /*!< are the optional value fields available */
} TWiMODLORAWAN_RX_Data;


/**
 * @brief Structure containing a received MAC command
 */
typedef struct TWiMODLORAWAN_RX_MacCmdData
{
    UINT8       Length;                                                         /*!< length of the MAC command / data */
    UINT8       MacCmdData[WiMODLORAWAN_APP_PAYLOAD_LEN];                       /*!< MAC command / data */

    // optional RX info
    UINT8       ChannelIndex;                                                   /*!< used channel index (@see TLoRaWAN_Channel_* definitions)*/
    UINT8       DataRateIndex;                                                  /*!< used data rate index (@see TLoRaWANDataRate)  */
    INT8        RSSI;                                                           /*!< RSSI value for the received packet */
    INT8        SNR;                                                            /*!< SNR value for the received packet  */
    UINT8       RxSlot;                                                         /*!< number of the rx slot that contained the messsage */
    bool        OptionalInfoAvaiable;                                           /*!< are the optional value fields available */
} TWiMODLORAWAN_RX_MacCmdData;

/**
 * @brief Structure containing data of the joined network indication
 */
typedef struct TWiMODLORAWAN_RX_JoinedNwkData
{
    UINT8       StatusFormat;                                                   /*!< Status/Format Field for this message */
    UINT32      DeviceAddress;                                                  /*!< (new) address for this device within network */

    // optional RX info
    UINT8       ChannelIndex;                                                   /*!< used channel index (@see TLoRaWAN_Channel_* definitions)*/
    UINT8       DataRateIndex;                                                  /*!< used data rate index (@see TLoRaWANDataRate)  */
    INT8        RSSI;                                                           /*!< RSSI value for the received packet */
    INT8        SNR;                                                            /*!< SNR value for the received packet  */
    UINT8       RxSlot;                                                         /*!< number of the rx slot that contained the messsage */
    bool        OptionalInfoAvaiable;                                           /*!< are the optional value fields available */
} TWiMODLORAWAN_RX_JoinedNwkData;

/**
 * @brief Radio stack configuration related to to LoRaWAN specific options.
 */
typedef struct TWiMODLORAWAN_RadioStackConfig
{
    UINT8       DataRateIndex;                                                  /*!< data rate to use (@see TLoRaWANDataRate) */
    UINT8       TXPowerLevel;                                                   /*!< default / initial power level settings to be used */
    UINT8       Options;                                                        /*!< misc Options (see LORAWAN_STK_OPTION defines) */
    UINT8       PowerSavingMode;                                                /*!< set automatic power saving mode */
    UINT8       Retransmissions;                                                /*!< max number of retransmissions (for C-Data) to use */
    UINT8       BandIndex;                                                      /*!< Frequency band to use (@see TLoRaWAN_FreqBand) */
} TWiMODLORAWAN_RadioStackConfig;

/**
 * @brief LoRaWAN MAC command structure
 */
typedef struct TWiMODLORAWAN_MacCmd
{
    UINT8       DataServiceType;                                                /*!< Data service type */ // see LORAWAN_MAC_DATA_SERVICE_TYPE
    UINT8       MacCmdID;                                                       /*!< MAC command ID    */ // see LoRaWAN Spec for command IDs
    UINT8       Payload[WiMODLORAWAN_MAC_CMD_PAYLOAD_LENGTH];                   /*!< payload for MAC command */
    UINT8       Length;                                                         /*!< length of payload */
} TWiMODLORAWAN_MacCmd;


/**
 * @brief Data structure for received Acknowledgment data
 */
typedef struct TWiMODLORAWAN_RX_ACK_Data
{
    UINT8       StatusFormat;                                                   /*!< Status/Format Field for this message */

    // optional RX info
    UINT8       ChannelIndex;                                                   /*!< used channel index (@see TLoRaWAN_Channel_* definitions)*/
    UINT8       DataRateIndex;                                                  /*!< used data rate index (@see TLoRaWANDataRate)  */
    INT8        RSSI;                                                           /*!< RSSI value for the received packet */
    INT8        SNR;                                                            /*!< SNR value for the received packet  */
    UINT8       RxSlot;                                                         /*!< number of the rx slot that contained the messsage */
    bool        OptionalInfoAvaiable;                                           /*!< are the optional value fields available */
} TWiMODLORAWAN_RX_ACK_Data;


/**
 * @brief Structure containing all data about LoRaWAN network status
 */
typedef struct TWiMODLORAWAN_NwkStatus_Data
{
    UINT8		NetworkStatus; 													/*!< Current NetworkStatus (@see TLoRaWAN_NwkStatus) */
    UINT32		DeviceAddress;													/*!< 32bit DeviceAddress of module in LoRa network */
    UINT8		DataRateIndex; 													/*!< used data rate index (@see TLoRaWANDataRate)  */
    UINT8		PowerLevel;				    									/*!< transmit power in dBm */
    UINT8		MaxPayloadSize;													/*!< maximum number of bytes allowed in payload field */
} TWiMODLORAWAN_NwkStatus_Data;


#endif /* ARDUINO_WIMOD_SAP_LORAWAN_IDS_H_ */
